<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\BranchModel;
use App\Models\ClientAddressModel;
use App\Models\ClientModel;
use App\Models\CompanyModel;
use App\Models\CountryModel;
use App\Models\RightsModel;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use function Laravel\Prompts\table;

class LoginController extends Controller
{
    public function showLoginForm()
    {
        if (auth()->user()) {
            return redirect('/dashboard');
        }

        $company = CompanyModel::first();

        return view('auth.login', [
            'company' => $company
        ]);
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email:rfc,dns',
            'password' => 'required',
        ]);

        $remember = $request->has('remember_me');

        if (Auth::attempt(['email' => $request->email, 'password' => $request->password], $remember)) {
            $user = User::where('email', $request->email)->first();
            Auth::login($user, $remember);

            return redirect()->intended('/setup');
        }

        return back()->withErrors([
            'email' => trans('auth.failed'),
        ])->withInput($request->only('email', 'remember_me'));
    }

    public function render()
    {
        return view('auth.login');
    }

    public function registrationForm()
    {
        $countries = CountryModel::get();
        $branch = BranchModel::get();
        return view('auth.registration-form', compact('countries', 'branch'));
    }
    public function clientRegister(Request $request)
    {

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'email' => 'required|email|unique:users,email',
            'contact_no' => 'required|string|max:15|unique:users,phone',
            'password' => 'required|string|min:6|confirmed',
            'postal_code' => 'required|string|max:10',
            'branch_id' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'gstin' => 'nullable|string|max:15',
            'profile_image' => 'nullable|image',
            'active_status' => 'required',
        ]);

        DB::beginTransaction();
        try {
            $right = RightsModel::where('rights_type', 'client')->first();
            $clientRight = $right ? $right->id : null;

            $user = User::create([
                'name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'email' => $validated['email'],
                'phone' => $validated['contact_no'],
                'password' => Hash::make($validated['password']),
                'user_flg' => 'C',
                'active_status' => $validated['active_status'],
                'branch_id' => $validated['branch_id'],
                'rights_id' => $clientRight
            ]);

            $profile_image_path = null;
            if ($request->hasFile('profile_image')) {
                $profile_image_path = storeFile($request->file('profile_image'), 'client');
            }

            $client = ClientModel::create([
                'user_id' => $user->id,
                'profile_img' => $profile_image_path,
                'website' => $request->input('website'),
                'postal_code' => $validated['postal_code'],
                'address' => $validated['address'],
                'note' => $validated['notes'],
                'gstin' => $validated['gstin'],
                'branch_id' => $validated['branch_id'],
                'active_status' => $validated['active_status'],
            ]);
            $allBranchIds = DB::table('branch')->pluck('id');
            $allCompanyIds = DB::table('company')->pluck('id');

            $branchInsertData = [];
            $comapanyInsertData = [];

            foreach ($allBranchIds as $branchId) {
                $branchInsertData[] = [
                    'user_id' => $user->id,
                    'branch_id' => $branchId,
                ];
            }

            foreach ($allCompanyIds as $companyId) {
                $comapanyInsertData[] = [
                    'user_id' => $user->id,
                    'company_id' => $companyId,
                ];
            }

            DB::table('branch_user')->insert($branchInsertData);
            DB::table('company_user')->insert($comapanyInsertData);
            
            ClientAddressModel::create([
                'branch_id' => $validated['branch_id'],
                'client_id' => $client->id,
                'address' => $validated['address'],
                'created_by' => $client->id,
            ]);

            DB::commit();
            return redirect()->route('login')->with('success', 'Client Registered successfully!');
        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Client Add' . $e);
            return back()->with('error', 'Something went wrong!')->withInput();
        }
    }
}